<?php

namespace App\Console\Commands;

use App\Models\Caja;
use App\Models\ExchangeRate;
use App\Services\CajaTasaConfigService;
use Illuminate\Console\Command;
use Carbon\Carbon;

class VerificarEstadoCajaTasa extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'cajatasa:verificar-estado {--empresa= : ID de empresa específica}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verificar el estado actual del sistema de cajas y tasas';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('=== ESTADO DEL SISTEMA DE CAJAS Y TASAS ===');
        $this->newLine();

        // Verificar configuración de horarios
        $this->verificarConfiguracionHorarios();
        
        // Verificar estado de tasas
        $this->verificarEstadoTasas();
        
        // Verificar estado de cajas
        $this->verificarEstadoCajas();
        
        // Verificar próximas acciones
        $this->verificarProximasAcciones();

        return Command::SUCCESS;
    }

    private function verificarConfiguracionHorarios(): void
    {
        $this->info('📅 CONFIGURACIÓN DE HORARIOS:');
        $this->line('Horarios de corte de caja: ' . implode(', ', CajaTasaConfigService::CORTE_CAJA_HORARIOS));
        $this->line('Período verificación tasa: ' . CajaTasaConfigService::TASA_VERIFICACION_INICIO . ' - ' . CajaTasaConfigService::TASA_VERIFICACION_FIN);
        $this->line('Actualización automática tasa: ' . CajaTasaConfigService::TASA_ACTUALIZACION_AUTO);
        
        $config = CajaTasaConfigService::getConfiguracionActual();
        $this->line('Estado actual:');
        $this->line('  - En verificación tasa: ' . ($config['en_verificacion_tasa'] ? '✅ SÍ' : '❌ NO'));
        $this->line('  - Es hora actualización tasa: ' . ($config['es_actualizacion_tasa'] ? '✅ SÍ' : '❌ NO'));
        $this->line('  - Es hora corte caja: ' . ($config['hora_corte_caja'] ? '✅ ' . $config['hora_corte_caja'] : '❌ NO'));
        $this->newLine();
    }

    private function verificarEstadoTasas(): void
    {
        $this->info('💰 ESTADO DE TASAS:');
        
        $tasaHoy = ExchangeRate::getTodayRate();
        
        if ($tasaHoy) {
            $this->line('✅ Tasa del día registrada:');
            $this->line("  - USD: {$tasaHoy->usd_rate} Bs.");
            $this->line("  - EUR: {$tasaHoy->eur_rate} Bs.");
            $this->line("  - Fuente: {$tasaHoy->source}");
            $this->line("  - Actualizada: {$tasaHoy->fetch_time}");
            
            $horasActualizacion = Carbon::parse($tasaHoy->fetch_time)->diffInHours(now());
            if ($horasActualizacion > 6) {
                $this->warn("  ⚠️  Tasa desactualizada (hace {$horasActualizacion} horas)");
            } else {
                $this->line("  ✅ Tasa actualizada hace {$horasActualizacion} horas");
            }
        } else {
            $this->warn('❌ No hay tasa registrada para hoy');
        }
        
        // Verificar últimas tasas
        $ultimasTasas = ExchangeRate::whereDate('date', '>=', now()->subDays(7))
            ->orderBy('date', 'desc')
            ->limit(5)
            ->get();
            
        if ($ultimasTasas->count() > 0) {
            $this->line("\n📊 Últimas tasas:");
            foreach ($ultimasTasas as $tasa) {
                $this->line("  {$tasa->date}: USD {$tasa->usd_rate} Bs. ({$tasa->source})");
            }
        }
        
        $this->newLine();
    }

    private function verificarEstadoCajas(): void
    {
        $this->info('💼 ESTADO DE CAJAS:');
        
        $empresaId = $this->option('empresa');
        
        if ($empresaId) {
            $this->verificarCajasPorEmpresa($empresaId);
        } else {
            // Verificar todas las empresas
            $empresas = \App\Models\Empresa::all();
            foreach ($empresas as $empresa) {
                $this->line("\n🏢 Empresa: {$empresa->nombre}");
                $this->verificarCajasPorEmpresa($empresa->id);
            }
        }
        
        $this->newLine();
    }

    private function verificarCajasPorEmpresa(int $empresaId): void
    {
        $sucursales = \App\Models\Sucursal::where('empresa_id', $empresaId)->get();
        
        foreach ($sucursales as $sucursal) {
            $cajaAbierta = Caja::obtenerCajaAbierta($empresaId, $sucursal->id);
            $cajaHoy = Caja::where('empresa_id', $empresaId)
                ->where('sucursal_id', $sucursal->id)
                ->whereDate('fecha', today())
                ->first();
            
            $this->line("  📍 Sucursal: {$sucursal->nombre}");
            
            if ($cajaAbierta) {
                $this->line("    ✅ Caja ABIERTA (ID: {$cajaAbierta->id})");
                $this->line("    💰 Monto inicial: {$cajaAbierta->monto_inicial} $");
                $this->line("    📅 Apertura: {$cajaAbierta->fecha_apertura}");
            } else {
                $this->warn("    ❌ No hay caja abierta");
            }
            
            if ($cajaHoy) {
                $this->line("    📊 Caja del día: {$cajaHoy->numero_corte}° corte");
                if ($cajaHoy->estado === 'cerrada') {
                    $this->line("    🔒 CERRADA ({$cajaHoy->fecha_cierre})");
                    $this->line("    💵 Monto final: {$cajaHoy->monto_final} $");
                }
            }
        }
    }

    private function verificarProximasAcciones(): void
    {
        $this->info('🔮 PRÓXIMAS ACCIONES:');
        
        $proximoCorte = CajaTasaConfigService::getProximoCorteCaja();
        if ($proximoCorte) {
            $this->line("🕐 Próximo corte de caja: {$proximoCorte->format('H:i')}");
            $this->line("   Faltan: " . now()->diffForHumans($proximoCorte, true));
        } else {
            $this->line("✅ No hay más cortes de caja programados para hoy");
        }
        
        if (CajaTasaConfigService::esHoraActualizacionTasa()) {
            $this->warn("🔄 ES HORA de actualización automática de tasa");
        } else {
            $this->line("⏳ Próxima actualización automática de tasa: 14:30");
        }
        
        $this->newLine();
        
        // Recomendaciones
        $this->info('💡 RECOMENDACIONES:');
        
        $tasaHoy = ExchangeRate::getTodayRate();
        if (!$tasaHoy) {
            $this->warn("   ⚠️  Registrar tasa del día");
        }
        
        $cajasSinAbrir = Caja::whereDate('fecha', today())->where('estado', 'cerrada')->count();
        if ($cajasSinAbrir > 0) {
            $this->warn("   ⚠️  Hay {$cajasSinAbrir} cajas cerradas que pueden necesitar reapertura");
        }
        
        $this->line("   ℹ️  Ejecutar 'php artisan cajatasa:automatizar --force' para forzar automatización");
        $this->line("   ℹ️  Ejecutar 'php artisan tasa:actualizar-programada --force' para forzar actualización de tasa");
        $this->line("   ℹ️  Ejecutar 'php artisan caja:corte-automatico --force' para forzar corte de caja");
    }
}