<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Dotenv\Dotenv;
use Illuminate\Support\Facades\Http;
use Exception;

class TestWhatsAppJWT extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'whatsapp:test-jwt {phone : Número de teléfono} {message : Mensaje a enviar}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Probar envío de mensajes de WhatsApp con autenticación JWT';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $phone = $this->argument('phone');
        $message = $this->argument('message');

        $this->info('Iniciando prueba de envío de WhatsApp con JWT...');

        // Cargar configuración desde el archivo .env del servicio
        $configFile = base_path('whatsapp-service-config.env');
        if (!file_exists($configFile)) {
            $this->error('No se encontró el archivo de configuración whatsapp-service-config.env');
            return 1;
        }

        // Parsear el archivo de configuración
        $config = $this->parseEnvFile($configFile);

        if (empty($config['LARAVEL_API_KEY'])) {
            $this->error('No se encontró el token JWT en la configuración');
            return 1;
        }

        $this->info('Token JWT encontrado: ' . substr($config['LARAVEL_API_KEY'], 0, 20) . '...');

        // Verificar el token primero
        $this->info('Verificando token JWT...');
        $verifyResult = $this->verifyJWTToken($config['LARAVEL_API_KEY']);

        if (!$verifyResult['valid']) {
            $this->error('Token JWT inválido: ' . $verifyResult['error']);
            return 1;
        }

        $this->info('✓ Token JWT válido');
        $this->info('Usuario: ' . $verifyResult['user']['name'] . ' (' . $verifyResult['user']['email'] . ')');

        // Intentar enviar el mensaje
        $this->info('Enviando mensaje a ' . $phone . '...');
        $result = $this->sendWhatsAppMessage($config, $phone, $message);

        if (!$result['success']) {
            $this->error('✗ Error al enviar el mensaje: ' . $result['error']);
            return 1;
        }
        
        // El mensaje de éxito ya se muestra en sendWhatsAppMessage

        return 0;
    }

    private function parseEnvFile($filePath)
    {
        $config = [];
        $lines = file($filePath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);

        foreach ($lines as $line) {
            if (strpos($line, '#') === 0) continue;
            if (strpos($line, '=') === false) continue;

            list($key, $value) = explode('=', $line, 2);
            $config[trim($key)] = trim($value);
        }

        return $config;
    }

    private function verifyJWTToken($token)
    {
        try {
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $token,
                'Accept' => 'application/json',
            ])->post('http://localhost:8000/api/verify-token');

            if ($response->successful()) {
                return [
                    'valid' => true,
                    'user' => $response->json('user')
                ];
            } else {
                return [
                    'valid' => false,
                    'error' => $response->json('error') ?? 'Error desconocido'
                ];
            }
        } catch (Exception $e) {
            return [
                'valid' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    private function sendWhatsAppMessage($config, $phone, $message)
    {
        try {
            $this->info("Enviando mensaje a {$phone}...");
            
            // Usar el servicio de WhatsApp directamente
            $whatsappService = new \App\Services\WhatsAppService();
            
            // Configurar el token JWT en el servicio
            if (!empty($config['LARAVEL_API_KEY'])) {
                $whatsappService->setJwtToken($config['LARAVEL_API_KEY']);
            }
            
            // Enviar el mensaje (el servicio ya maneja el modo simulado)
            $result = $whatsappService->sendMessage($phone, $message);
            
            if ($result['success']) {
                if (isset($result['simulated']) && $result['simulated']) {
                    $this->info('✓ Mensaje enviado exitosamente (SIMULADO)');
                    $this->info('Nota: ' . ($result['note'] ?? 'Este es un envío simulado'));
                } else {
                    $this->info('✓ Mensaje enviado exitosamente');
                }
                
                if (isset($result['message_id'])) {
                    $this->info('ID del mensaje: ' . $result['message_id']);
                }
                
                return [
                    'success' => true,
                    'message_id' => $result['message_id'] ?? null,
                    'simulated' => $result['simulated'] ?? false
                ];
            } else {
                $error = $result['error'] ?? 'Error desconocido';
                $this->error("✗ Error al enviar el mensaje: " . $error);
                return [
                    'success' => false,
                    'error' => $error
                ];
            }
        } catch (Exception $e) {
            $this->error("✗ Excepción al enviar mensaje: " . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}