<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;

class TestWelcomeWhatsAppSincrono extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:welcome-whatsapp-sincrono';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Prueba de mensaje de bienvenida WhatsApp sincrónico con el número 04242115948';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== PRUEBA DE MENSAJE DE BIENVENIDA WHATSAPP SINCRÓNICO ===');
        
        // Número de prueba
        $telefonoPrueba = '04242115948';
        $usernamePrueba = 'test_cliente_' . time();
        $emailPrueba = 'test_' . time() . '@ejemplo.com';
        
        $this->info("Creando cliente de prueba...");
        $this->info("Teléfono: $telefonoPrueba");
        $this->info("Usuario: $usernamePrueba");
        $this->info("Email: $emailPrueba");
        
        try {
            DB::beginTransaction();
            
            // Crear usuario de prueba
            $user = new User();
            $user->name = 'Cliente Prueba WhatsApp';
            $user->email = $emailPrueba;
            $user->username = $usernamePrueba;
            $user->telefono = $telefonoPrueba;
            $user->password = Hash::make('test123456');
            $user->empresa_id = 1;
            $user->sucursal_id = 1;
            $user->email_verified_at = now();
            $user->status = 1;
            $user->save();
            $user->assignRole('Cliente');
            
            DB::commit();
            
            $this->info("✅ Cliente creado exitosamente!");
            $this->info("ID: {$user->id}");
            $this->info("Nombre: {$user->name}");
            $this->info("Teléfono original: {$user->telefono}");
            
            // Ahora simulamos el proceso de registro como lo haría el componente Index
            $this->info("\n=== PROBANDO ENVÍO SINCRÓNICO DE MENSAJE ===");
            
            // Generar token temporal
            $credentialToken = \App\Services\TemporaryTokenService::generateCredentialToken(
                $user->id, $user->username, 'test123456'
            );
            
            $this->info("Token generado: {$credentialToken}");
            
            // Probar el formateo de número
            $whatsappService = new \App\Services\WhatsAppService();
            $telefonoFormateado = $whatsappService->cleanPhoneNumber($user->telefono);
            
            $this->info("Teléfono formateado: {$telefonoFormateado}");
            
            // Crear mensaje de bienvenida
            $message = "🎉 *¡Bienvenido {$user->name}!*\n\n";
            $message .= "Gracias por registrarte en nuestro sistema de pedidos.\n\n";
            $message .= "📱 *Tus credenciales de acceso:*\n";
            $message .= "Usuario: {$user->username}\n";
            $message .= "Contraseña: test123456\n";
            $message .= "Token de acceso: {$credentialToken}\n\n";
            $message .= "Ingresa a nuestro sistema y realiza tus pedidos de manera fácil y rápida.\n\n";
            $message .= "¿Necesitas ayuda? No dudes en contactarnos.\n\n";
            $message .= "¡Bienvenido a bordo! 🚀";
            
            $this->info("\n=== MENSAJE A ENVIAR ===");
            $this->info($message);
            
            // Enviar mensaje
            $this->info("\n=== ENVIANDO MENSAJE ===");
            $result = $whatsappService->sendMessage($telefonoFormateado, $message);
            
            if ($result && ($result['success'] || isset($result['simulated']))) {
                $this->info("✅ MENSAJE ENVIADO EXITOSAMENTE!");
                $this->info("Respuesta del servicio: " . json_encode($result));
                
                Log::info('Mensaje de bienvenida enviado exitosamente en prueba', [
                    'user_id' => $user->id,
                    'telefono' => $telefonoFormateado,
                    'result' => $result
                ]);
            } else {
                $this->error("❌ ERROR AL ENVIAR MENSAJE");
                $this->error("Respuesta: " . json_encode($result));
                
                Log::warning('No se pudo enviar mensaje de bienvenida en prueba', [
                    'user_id' => $user->id,
                    'telefono' => $telefonoFormateado,
                    'result' => $result
                ]);
            }
            
            // Limpiar usuario de prueba
            $this->info("\n=== LIMPIEZA ===");
            $user->delete();
            $this->info("Usuario de prueba eliminado.");
            
        } catch (\Exception $e) {
            DB::rollback();
            $this->error("❌ ERROR EN LA PRUEBA: " . $e->getMessage());
            Log::error('Error en prueba de mensaje de bienvenida sincrónico', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
        }
        
        $this->info("\n=== PRUEBA FINALIZADA ===");
    }
}