<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Pedido;
use App\Services\WhatsAppService;
use Illuminate\Support\Facades\Log;

class TestOrderWhatsAppNotification extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:order-whatsapp-notification {pedido_id?}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Probar el envío de notificación WhatsApp para un pedido específico o el último pedido';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🧪 Iniciando prueba de notificación WhatsApp para pedido...');
        
        try {
            // Obtener el pedido a probar
            $pedidoId = $this->argument('pedido_id');
            
            if ($pedidoId) {
                $pedido = Pedido::with(['user.empresa.pais', 'detalles.producto'])->find($pedidoId);
                if (!$pedido) {
                    $this->error("❌ No se encontró el pedido con ID: {$pedidoId}");
                    return 1;
                }
            } else {
                // Obtener el último pedido
                $pedido = Pedido::with(['user.empresa.pais', 'detalles.producto'])->latest()->first();
                if (!$pedido) {
                    $this->error('❌ No se encontraron pedidos en la base de datos');
                    return 1;
                }
            }
            
            $this->info("📋 Pedido seleccionado:");
            $this->info("• ID: {$pedido->id}");
            $this->info("• Código: #{$pedido->codigo}");
            $this->info("• Usuario: {$pedido->user->name}");
            $this->info("• Teléfono: {$pedido->user->telefono}");
            $this->info("• Total: " . ($pedido->user->empresa && $pedido->user->empresa->moneda === 'USD' ? '$' : 'Bs. ') . number_format($pedido->total_usd, 2));
            $this->info("• Fecha: {$pedido->created_at->format('d/m/Y H:i')}");
            
            // Verificar que el usuario tenga teléfono
            if (!$pedido->user->telefono) {
                $this->error('❌ El usuario no tiene número de teléfono registrado');
                return 1;
            }
            
            // Mostrar detalles del pedido
            if ($pedido->detalles && $pedido->detalles->count() > 0) {
                $this->info("\n🛍️ Productos del pedido:");
                foreach ($pedido->detalles as $detalle) {
                    $productoNombre = $detalle->product_name ?? $detalle->producto->name ?? 'Producto';
                    $this->info("• {$productoNombre} x{$detalle->quantity}");
                }
            }
            
            // Probar el servicio de WhatsApp
            $this->info("\n📱 Probando conexión con WhatsApp API...");
            $whatsappService = new WhatsAppService();
            
            $status = $whatsappService->getStatus();
            if ($status && isset($status['success']) && $status['success']) {
                $this->info('✅ WhatsApp API está conectada');
                $this->info('• Estado: ' . ($status['isReady'] ?? false ? 'Listo' : 'No listo'));
            } else {
                $this->warn('⚠️ WhatsApp API no está disponible o no está conectada');
            }
            
            // Formatear el mensaje para mostrarlo
            $this->info("\n📝 Mensaje que se enviará:");
            $this->info("========================================");
            $message = $this->formatOrderMessage($pedido);
            $this->line($message);
            $this->info("========================================");
            
            // Preguntar si desea enviar el mensaje
            if ($this->confirm('¿Deseas enviar este mensaje WhatsApp al cliente?')) {
                $this->info("\n🚀 Enviando notificación WhatsApp...");
                
                $result = $whatsappService->sendOrderNotification($pedido);
                
                if ($result && isset($result['success']) && $result['success']) {
                    $this->info('✅ Notificación WhatsApp enviada exitosamente');
                    if (isset($result['messageId'])) {
                        $this->info('• ID del mensaje: ' . $result['messageId']);
                    }
                } else {
                    $this->error('❌ Error al enviar la notificación');
                    if (isset($result['error'])) {
                        $this->error('• Detalles: ' . $result['error']);
                    }
                }
            } else {
                $this->info('⏭️ Envío cancelado por el usuario');
            }
            
            $this->info('\n✅ Prueba finalizada');
            return 0;
            
        } catch (\Exception $e) {
            $this->error('❌ Error durante la prueba: ' . $e->getMessage());
            Log::error('Error en TestOrderWhatsAppNotification: ' . $e->getMessage());
            return 1;
        }
    }
    
    /**
     * Formatear mensaje de notificación de pedido (copia del método del servicio)
     *
     * @param \App\Models\Pedido $pedido
     * @return string
     */
    private function formatOrderMessage($pedido): string
    {
        $user = $pedido->user;
        $empresa = $user->empresa ?? null;
        
        $message = "📦 *¡Pedido Confirmado!* 📦\n\n";
        $message .= "*Hola {$user->name}!*\n\n";
        $message .= "✅ Tu pedido *#{$pedido->codigo}* ha sido recibido y está siendo procesado.\n\n";
        
        // Detalles del pedido
        $message .= "📋 *Detalles del Pedido:*\n";
        $message .= "• Código: #{$pedido->codigo}\n";
        $message .= "• Fecha: " . $pedido->created_at->format('d/m/Y H:i') . "\n";
        $message .= "• Total: " . ($empresa && $empresa->moneda === 'USD' ? '$' . number_format($pedido->total_usd, 2) : 'Bs. ' . number_format($pedido->total_bs, 2)) . "\n";
        $message .= "• Estado: {$pedido->estado}\n";
        $message .= "• Método de Pago: {$pedido->metodo_pago}\n\n";
        
        // Productos
        if ($pedido->detalles && $pedido->detalles->count() > 0) {
            $message .= "🛍️ *Productos:*\n";
            foreach ($pedido->detalles as $detalle) {
                $productoNombre = $detalle->product_name ?? $detalle->producto->name ?? 'Producto';
                $message .= "• {$productoNombre} x{$detalle->quantity} = " . 
                    ($empresa && $empresa->moneda === 'USD' ? '$' . number_format($detalle->total_usd, 2) : 'Bs. ' . number_format($detalle->total_bs, 2)) . "\n";
            }
            $message .= "\n";
        }
        
        // Información de entrega
        if ($pedido->ubicacion) {
            $message .= "📍 *Dirección de Entrega:*\n{$pedido->ubicacion}\n\n";
        }
        
        // Próximos pasos
        $message .= "⏳ *Próximos Pasos:*\n";
        $message .= "• Revisaremos tu pedido\n";
        $message .= "• Confirmaremos la disponibilidad de los productos\n";
        $message .= "• Te notificaremos cuando esté listo para entrega\n\n";
        
        $message .= "¡Gracias por tu compra! 😊\n\n";
        
        if ($empresa) {
            $message .= "Atentamente,\n{$empresa->nombre}\n";
            if ($empresa->telefono) {
                $message .= "📞 {$empresa->telefono}\n";
            }
        }
        
        return $message;
    }
}