<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Empresa;
use App\Services\WhatsAppService;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\DB;

class TestEmpresaCreation extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'test:empresa-creation';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Probar la creación de empresa con token de WhatsApp';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Creando empresa de prueba...');

        try {
            // Generar token único de WhatsApp
            $whatsappApiKey = $this->generateWhatsAppApiKey('Empresa de Prueba S.A.');

            $empresa = Empresa::create([
                'razon_social' => 'Empresa de Prueba S.A.',
                'documento' => 'J-' . rand(10000000, 99999999) . '-' . rand(1, 9),
                'direccion' => 'Av. Principal, Edificio Test',
                'latitud' => 10.5,
                'longitud' => -66.9,
                'representante_legal' => 'Juan Pérez',
                'status' => true,
                'telefono' => '0212-' . rand(1000000, 9999999),
                'email' => 'contacto' . rand(100, 999) . '@empresaprueba.com',
                'pais_id' => 1,
                'whatsapp_api_key' => $whatsappApiKey,
            ]);

            $this->info("Empresa creada exitosamente!");
            $this->info("ID: {$empresa->id}");
            $this->info("Razón Social: {$empresa->razon_social}");
            $this->info("WhatsApp API Key: {$empresa->whatsapp_api_key}");

            // Registrar empresa en la base de datos del servicio WhatsApp
            $this->info('Registrando empresa en el servicio de WhatsApp...');
            
            try {
                DB::connection('mysql')->statement('
                    INSERT INTO larawhatsapp.companies (name, apiKey, rateLimitPerMinute, isActive, createdAt, updatedAt) 
                    VALUES (?, ?, 60, 1, NOW(), NOW())
                ', [$empresa->razon_social, $whatsappApiKey]);
                
                $this->info('✅ Empresa registrada exitosamente en el servicio de WhatsApp');
                
                // Verificar conexión con el servicio de WhatsApp
                $this->info('Verificando conexión con el servicio de WhatsApp...');
                $whatsAppService = new WhatsAppService();
                
                try {
                    $status = $whatsAppService->getStatus($whatsappApiKey);
                    
                    if ($status && isset($status['success']) && $status['success']) {
                        $this->info('✅ API Key válida y servicio de WhatsApp accesible');
                        $this->info("Estado de conexión: {$status['connectionState']}");
                        if (isset($status['company'])) {
                            $this->info("Empresa configurada: {$status['company']}");
                        }
                    } else {
                        $this->warn('⚠️  API Key generada, pero el servicio de WhatsApp no está accesible');
                        if ($status && isset($status['error'])) {
                            $this->warn("Error: {$status['error']}");
                        }
                    }
                } catch (\Exception $e) {
                    $this->error('❌ Error al verificar conexión con WhatsApp: ' . $e->getMessage());
                }
                
            } catch (\Exception $e) {
                $this->error('❌ Error al registrar empresa en WhatsApp: ' . $e->getMessage());
                $this->warn('⚠️  La empresa fue creada en Laravel, pero no se pudo registrar en WhatsApp');
            }

            // Probar el token con el servicio
            $this->info('Probando conexión con el servicio...');
            $status = $whatsAppService->getStatus($whatsappApiKey);
            
            if ($status) {
                $this->info('✅ Conexión exitosa con el servicio WhatsApp');
                $this->info("Estado: {$status['connectionState']}");
                $this->info("Empresa: {$status['company']}");
            } else {
                $this->warn('⚠️  No se pudo obtener el estado del servicio');
            }

        } catch (\Exception $e) {
            $this->error('Error al crear empresa: ' . $e->getMessage());
            return 1;
        }

        return 0;
    }

    /**
     * Genera un token único de WhatsApp para la empresa
     */
    private function generateWhatsAppApiKey(string $companyName): string
    {
        $prefix = 'wapi_';
        $companyNameClean = str_replace(' ', '-', strtolower($companyName));
        $companyNameClean = preg_replace('/[^a-z0-9\-]/', '', $companyNameClean);
        $randomString = Str::random(16);
        $timestamp = now()->format('YmdHis');
        
        return $prefix . $companyNameClean . '_' . $timestamp . '_' . $randomString;
    }
}