<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Database\Seeders\EmpleadosTableSeeder;
use Illuminate\Support\Facades\Artisan;

class SeedEmpleadosCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'seed:empleados {--fresh : Ejecutar fresh antes de seedear}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Ejecutar el seeder de empleados con usuarios relacionados';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Iniciando proceso de seed de empleados...');

        // Si se especifica --fresh, ejecutar migrate:fresh
        if ($this->option('fresh')) {
            $this->warn('⚠️  Ejecutando migrate:fresh...');
            Artisan::call('migrate:fresh');
            $this->info('✅ Base de datos reiniciada');

            // Ejecutar los seeders necesarios primero
            $this->info('🔄 Ejecutando seeders base...');
            Artisan::call('db:seed', ['--class' => 'Database\Seeders\RolesAndPermissionsSeeder']);
            Artisan::call('db:seed', ['--class' => 'Database\Seeders\PaisSeeder']);
            Artisan::call('db:seed', ['--class' => 'Database\Seeders\EmpresaSeeder']);
            Artisan::call('db:seed', ['--class' => 'Database\Seeders\SucursalSeeder']);
            $this->info('✅ Seeders base ejecutados');
        }

        // Confirmar antes de ejecutar
        if (!$this->option('fresh') && !$this->confirm('¿Deseas ejecutar el seeder de empleados? Esto creará usuarios y empleados de prueba.')) {
            $this->info('❌ Operación cancelada');
            return;
        }

        try {
            // Ejecutar el seeder de empleados
            $this->info('👥 Ejecutando seeder de empleados...');
            $seeder = new EmpleadosTableSeeder();
            $seeder->run();

            $this->info('✅ Seeder de empleados ejecutado exitosamente!');

            // Mostrar resumen
            $this->table(
                ['Información', 'Detalle'],
                [
                    ['Contraseña por defecto', 'password123'],
                    ['Rol asignado', 'Empleado'],
                    ['Empresa', 'Empresa Demo (o primera existente)'],
                    ['Sucursal', 'Sucursal Principal (o primera existente)'],
                    ['Total empleados creados', '5'],
                    ['Total usuarios creados', '5']
                ]
            );

            $this->newLine();
            $this->info('📧 Credenciales de acceso para los empleados:');
            $this->line('Los empleados pueden iniciar sesión con:');
            $this->line('- Email: [correo_del_empleado]');
            $this->line('- Contraseña: password123');

            $this->newLine();
            $this->info('🔄 Para cambiar contraseñas, ejecuta:');
            $this->line('php artisan tinker');
            $this->line('>>> $user = App\\Models\\User::find(1);');
            $this->line('>>> $user->password = bcrypt("nueva_contraseña");');
            $this->line('>>> $user->save();');

        } catch (\Exception $e) {
            $this->error('❌ Error al ejecutar el seeder: ' . $e->getMessage());
            $this->error('📋 Trace: ' . $e->getTraceAsString());
            return 1;
        }

        return 0;
    }
}
