<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\MensajePendiente;
use App\Services\PedidoService;

class ProcesarMensajesPendientes extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'whatsapp:procesar-pendientes 
                            {--max=50 : Número máximo de mensajes a procesar}
                            {--tipo= : Procesar solo mensajes de un tipo específico}
                            {--force : Forzar el procesamiento incluso si hay pocos mensajes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Procesar mensajes de WhatsApp pendientes';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🚀 Iniciando procesamiento de mensajes pendientes...');
        
        $maxMensajes = (int) $this->option('max');
        $tipo = $this->option('tipo');
        $force = $this->option('force');
        
        // Obtener mensajes pendientes
        $query = MensajePendiente::pendientes()
            ->where('intentos', '<', 5) // Máximo 5 intentos
            ->where(function ($query) {
                // No reintentar mensajes que fallaron hace menos de 5 minutos
                $query->whereNull('ultimo_intento')
                      ->orWhere('ultimo_intento', '<', now()->subMinutes(5));
            });
        
        if ($tipo) {
            $query->porTipo($tipo);
        }
        
        $mensajes = $query->orderBy('created_at', 'asc')
                          ->limit($maxMensajes)
                          ->get();
        
        if ($mensajes->isEmpty()) {
            $this->info('✅ No hay mensajes pendientes para procesar.');
            return Command::SUCCESS;
        }
        
        $this->info("📱 Se encontraron {$mensajes->count()} mensajes pendientes");
        
        $procesados = 0;
        $exitosos = 0;
        $fallidos = 0;
        
        $progressBar = $this->output->createProgressBar($mensajes->count());
        $progressBar->start();
        
        foreach ($mensajes as $mensaje) {
            try {
                // Reintentar el envío
                $resultado = $this->reintentarEnvio($mensaje);
                
                if ($resultado['exito']) {
                    $exitosos++;
                    $mensaje->marcarComoEnviado();
                    $this->info("✅ Mensaje enviado: {$mensaje->telefono} (Intento #{$mensaje->intentos})");
                } else {
                    $fallidos++;
                    $mensaje->marcarComoFallido($resultado['error']);
                    $this->error("❌ Mensaje falló: {$mensaje->telefono} - {$resultado['error']}");
                }
                
                $procesados++;
                $progressBar->advance();
                
                // Pequeña pausa entre mensajes para no sobrecargar el servicio
                usleep(100000); // 0.1 segundos
                
            } catch (\Exception $e) {
                $fallidos++;
                $mensaje->marcarComoFallido($e->getMessage());
                $this->error("❌ Error procesando mensaje: " . $e->getMessage());
                $progressBar->advance();
            }
        }
        
        $progressBar->finish();
        $this->newLine();
        
        // Resumen final
        $this->info("\n📊 Resumen del procesamiento:");
        $this->info("✅ Procesados: {$procesados}");
        $this->info("✅ Exitosos: {$exitosos}");
        $this->info("❌ Fallidos: {$fallidos}");
        
        // Verificar si hay muchos mensajes pendientes
        $pendientesRestantes = MensajePendiente::pendientes()->count();
        if ($pendientesRestantes > 100) {
            $this->warn("⚠️  Aún quedan {$pendientesRestantes} mensajes pendientes. Considera ejecutar el comando nuevamente.");
        }
        
        $this->info('🎉 Procesamiento completado.');
        
        return Command::SUCCESS;
    }
    
    /**
     * Reintentar el envío de un mensaje
     */
    private function reintentarEnvio(MensajePendiente $mensaje): array
    {
        try {
            // Obtener el servicio de PedidoService
            $pedidoService = app(PedidoService::class);
            
            // Reintentar con el método de reintento
            $resultado = $pedidoService->enviarWhatsAppConReintento(
                $mensaje->telefono,
                $mensaje->mensaje,
                3, // Máximo 3 intentos adicionales
                2  // 2 segundos entre intentos
            );
            
            return $resultado;
            
        } catch (\Exception $e) {
            return [
                'exito' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}