<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\DetallePedido;

class FixDetallesPedidoNull extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'pedidos:fix-null-cantidad';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Corregir detalles de pedido con cantidad NULL';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('=== ANÁLISIS DE DETALLES DE PEDIDO CON CANTIDAD NULL ===');
        
        // Buscar todos los detalles con cantidad NULL
        $detallesConNull = DetallePedido::whereNull('cantidad')->get();
        
        $this->info("Detalles de pedido con cantidad NULL encontrados: " . $detallesConNull->count());
        
        if ($detallesConNull->count() > 0) {
            $this->info('');
            $this->info('Detalles afectados:');
            $this->table(
                ['ID', 'Pedido ID', 'Producto ID', 'Precio Unitario', 'Subtotal'],
                $detallesConNull->map(function ($detalle) {
                    return [
                        $detalle->id,
                        $detalle->pedido_id,
                        $detalle->producto_id,
                        $detalle->precio_unitario,
                        $detalle->subtotal
                    ];
                })
            );
            
            if ($this->confirm('¿Desea corregir estos detalles estableciendo cantidad = 1?')) {
                $this->info('');
                $this->info('Corrigiendo detalles...');
                $corregidos = 0;
                
                foreach ($detallesConNull as $detalle) {
                    // Calcular la cantidad basada en el subtotal y precio unitario
                    if ($detalle->precio_unitario > 0 && $detalle->subtotal > 0) {
                        $cantidadCalculada = $detalle->subtotal / $detalle->precio_unitario;
                        $cantidad = round($cantidadCalculada, 0); // Redondear a entero
                        
                        if ($cantidad <= 0) {
                            $cantidad = 1; // Valor por defecto
                        }
                    } else {
                        $cantidad = 1; // Valor por defecto
                    }
                    
                    $detalle->cantidad = $cantidad;
                    $detalle->save();
                    $corregidos++;
                    
                    $this->info("Detalle ID {$detalle->id}: cantidad establecida a {$cantidad}");
                }
                
                $this->info("");
                $this->info("✓ Se corrigieron {$corregidos} detalles de pedido.");
                
                // Verificar si aún hay detalles con cantidad NULL
                $detallesRestantes = DetallePedido::whereNull('cantidad')->count();
                if ($detallesRestantes > 0) {
                    $this->warn("⚠ Aún quedan {$detallesRestantes} detalles con cantidad NULL.");
                } else {
                    $this->info("✓ Todos los detalles ahora tienen cantidad válida.");
                }
                
            } else {
                $this->info('Operación cancelada.');
            }
        } else {
            $this->info('✓ No se encontraron detalles de pedido con cantidad NULL.');
        }
        
        $this->info('=== FIN DEL ANÁLISIS ===');
        
        return Command::SUCCESS;
    }
}