<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\WhatsAppService;

class DebugDashboardPerformance extends Command
{
    protected $signature = 'whatsapp:debug-performance';
    protected $description = 'Depurar el rendimiento del Dashboard de WhatsApp';

    public function handle()
    {
        $this->info('=== Analizando rendimiento del Dashboard ===');
        
        $startTime = microtime(true);
        $whatsappService = new WhatsAppService();
        $serviceInitTime = microtime(true);
        
        $this->info('1. Tiempo de inicialización del servicio: ' . round(($serviceInitTime - $startTime) * 1000, 2) . 'ms');
        
        // Medir testConnection
        $testStart = microtime(true);
        $connectionTest = $whatsappService->testConnection();
        $testEnd = microtime(true);
        $this->info('2. Tiempo de testConnection: ' . round(($testEnd - $testStart) * 1000, 2) . 'ms');
        $this->info('   Resultado: ' . ($connectionTest['success'] ? 'Conectado' : 'Falló'));
        
        // Medir getStatus
        $statusStart = microtime(true);
        $statusResult = $whatsappService->getStatus();
        $statusEnd = microtime(true);
        $this->info('3. Tiempo de getStatus: ' . round(($statusEnd - $statusStart) * 1000, 2) . 'ms');
        $this->info('   Estado: ' . ($statusResult['status'] ?? 'desconocido'));
        
        // Medir getDetailedStatus
        $detailedStart = microtime(true);
        $detailedStatus = $whatsappService->getDetailedStatus();
        $detailedEnd = microtime(true);
        $this->info('4. Tiempo de getDetailedStatus: ' . round(($detailedEnd - $detailedStart) * 1000, 2) . 'ms');
        
        // Medir getQueueStats
        $queueStart = microtime(true);
        $queueStats = $whatsappService->getQueueStats();
        $queueEnd = microtime(true);
        $this->info('5. Tiempo de getQueueStats: ' . round(($queueEnd - $queueStart) * 1000, 2) . 'ms');
        
        // Medir getDailyStats
        $dailyStart = microtime(true);
        $dailyStats = $whatsappService->getDailyStats();
        $dailyEnd = microtime(true);
        $this->info('6. Tiempo de getDailyStats: ' . round(($dailyEnd - $dailyStart) * 1000, 2) . 'ms');
        
        $totalTime = microtime(true) - $startTime;
        $this->info('7. Tiempo total de operaciones: ' . round($totalTime * 1000, 2) . 'ms');
        
        // Análisis de posibles problemas
        $this->info('');
        $this->info('=== Análisis de rendimiento ===');
        
        if ($totalTime > 5) {
            $this->warn('⚠️  Tiempo total excesivo (>5s)');
            
            if (($statusEnd - $statusStart) > 2) {
                $this->warn('   - getStatus() es muy lento');
            }
            if (($detailedEnd - $detailedStart) > 1) {
                $this->warn('   - getDetailedStatus() es lento');
            }
            if (($queueEnd - $queueStart) > 1) {
                $this->warn('   - getQueueStats() es lento');
            }
            if (($dailyEnd - $dailyStart) > 1) {
                $this->warn('   - getDailyStats() es lento');
            }
        } else {
            $this->info('✅ Rendimiento aceptable');
        }
        
        $this->info('');
        $this->info('=== Recomendaciones ===');
        
        if ($totalTime > 3) {
            $this->info('💡 Sugerencias para mejorar el rendimiento:');
            $this->info('   1. Implementar caché para las llamadas a la API');
            $this->info('   2. Cargar estadísticas asíncronamente con Livewire');
            $this->info('   3. Reducir el timeout de las llamadas cURL');
            $this->info('   4. Implementar carga diferida (lazy loading)');
        }
    }
}