<?php

namespace App\Console\Commands;

use App\Services\CajaTasaAutomationService;
use App\Services\CajaTasaConfigService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CorteCajaAutomatico extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'caja:corte-automatico {--force : Forzar corte sin verificar horario}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Realizar corte de caja automático según horarios programados (1:00 PM y 6:00 PM)';

    /**
     * Execute the console command.
     */
    public function handle(): int
    {
        $this->info('Verificando corte automático de caja...');

        try {
            $force = $this->option('force');
            $horaCorte = CajaTasaConfigService::esHoraCorteCaja();
            
            if (!$force && !$horaCorte) {
                $this->warn('No es hora de corte automático (debe ser 1:00 PM o 6:00 PM)');
                $proximo = CajaTasaConfigService::getProximoCorteCaja();
                if ($proximo) {
                    $this->warn("Próximo corte: {$proximo->format('H:i')}");
                }
                $this->warn('Use --force para ejecutar de todas formas');
                return Command::SUCCESS;
            }

            $service = new CajaTasaAutomationService();
            $resultado = $service->realizarCorteCajaAutomatico($horaCorte ?? 'force');

            if ($resultado['exitoso']) {
                $this->info("✓ Corte de caja realizado exitosamente");
                $this->info("  {$resultado['mensaje']}");
                
                foreach ($resultado['cortes_realizados'] as $corte) {
                    $this->line("  - {$corte['empresa']} / {$corte['sucursal']}: {$corte['monto_final']}$");
                }
                
                Log::info('Corte de caja automático completado', $resultado);
            } else {
                $this->error("✗ Error: {$resultado['mensaje']}");
                Log::error('Error en corte de caja automático', $resultado);
            }

            return $resultado['exitoso'] ? Command::SUCCESS : Command::FAILURE;

        } catch (\Exception $e) {
            $this->error('Error crítico: ' . $e->getMessage());
            Log::error('Error crítico en corte de caja', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            return Command::FAILURE;
        }
    }
}