<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Services\WhatsAppService;

class CheckWhatsAppDashboard extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'whatsapp:check-dashboard';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verificar el estado completo del dashboard de WhatsApp';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('🔍 Verificando el dashboard de WhatsApp...');
        
        // Verificar que el servicio esté disponible
        $whatsappService = new WhatsAppService();
        
        $this->info('📱 Verificando conectividad con la API de WhatsApp...');
        $connection = $whatsappService->testConnection();
        
        if ($connection['success']) {
            $this->info('✅ Conexión con API de WhatsApp: EXITOSA');
            
            // Verificar estado
            $this->info('📊 Verificando estado del servicio...');
            $status = $whatsappService->getStatus();
            
            if ($status['success']) {
                $this->info('✅ Estado del servicio: ' . $status['status']);
                $this->info('   Listo: ' . ($status['isReady'] ? 'Sí' : 'No'));
                
                if (isset($status['qr']) && $status['qr']) {
                    $this->info('📱 Código QR disponible');
                }
            } else {
                $this->warn('⚠️  No se pudo obtener el estado: ' . ($status['error'] ?? 'Error desconocido'));
            }
            
            // Verificar estadísticas
            $this->info('📈 Verificando estadísticas...');
            $queueStats = $whatsappService->getQueueStats();
            $dailyStats = $whatsappService->getDailyStats();
            
            if ($queueStats['success']) {
                $this->info('✅ Estadísticas de cola disponibles');
                $stats = $queueStats['stats'] ?? [];
                $this->info('   Pendientes: ' . ($stats['pending'] ?? 0));
                $this->info('   Enviados: ' . ($stats['sent'] ?? 0));
                $this->info('   Fallidos: ' . ($stats['failed'] ?? 0));
            } else {
                $this->warn('⚠️  No se pudieron obtener estadísticas de cola');
            }
            
            if ($dailyStats['success']) {
                $this->info('✅ Estadísticas diarias disponibles');
                $stats = $dailyStats['stats'] ?? [];
                $this->info('   Hoy: ' . ($stats['sent'] ?? 0) . ' enviados, ' . 
                          ($stats['delivered'] ?? 0) . ' entregados, ' . 
                          ($stats['pending'] ?? 0) . ' pendientes, ' . 
                          ($stats['failed'] ?? 0) . ' fallidos');
            } else {
                $this->warn('⚠️  No se pudieron obtener estadísticas diarias');
            }
            
            // Verificar estado detallado
            $this->info('🔍 Verificando estado detallado...');
            $detailedStatus = $whatsappService->getDetailedStatus();
            
            if ($detailedStatus['success']) {
                $this->info('✅ Estado detallado disponible');
                if (isset($detailedStatus['detailed']) && !empty($detailedStatus['detailed'])) {
                    $this->info('   Información adicional disponible');
                }
            } else {
                $this->warn('⚠️  No se pudo obtener estado detallado');
            }
            
        } else {
            $this->error('❌ Conexión con API de WhatsApp: FALLIDA');
            $this->error('   Error: ' . ($connection['error'] ?? 'Error desconocido'));
            $this->warn('💡 El servicio funcionará en modo simulado');
        }
        
        $this->info('');
        $this->info('🎯 Resumen del Dashboard:');
        $this->info('   - Ruta: ' . route('admin.whatsapp.dashboard'));
        $this->info('   - Componente: App\Livewire\Admin\WhatsApp\Dashboard');
        $this->info('   - Servicio: App\Services\WhatsAppService');
        $this->info('   - Estado: ' . ($connection['success'] ? 'Funcional' : 'Modo simulado'));
        
        $this->info('');
        $this->info('✅ Verificación completada exitosamente!');
        $this->info('💡 Puedes acceder al dashboard en: ' . route('admin.whatsapp.dashboard'));
        
        return Command::SUCCESS;
    }
}