<?php

namespace App\Console\Commands;

use App\Models\Oferta;
use App\Models\User;
use App\Jobs\SendAutomaticNotifications;
use Illuminate\Console\Command;
use Carbon\Carbon;

class CheckExpiringOffers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'ofertas:check-expiring {--days=7 : Número de días para considerar ofertas por vencer}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Verifica ofertas por vencer y envía notificaciones automáticas';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $days = $this->option('days');
        $fecha_limite = Carbon::now()->addDays($days);

        $this->info("Buscando ofertas que vencen antes de: {$fecha_limite->format('Y-m-d')}");

        // Buscar ofertas por vencer
        $ofertas_por_vencer = Oferta::where('activo', true)
            ->where('fecha_fin', '<=', $fecha_limite)
            ->where('fecha_fin', '>=', Carbon::now())
            ->whereNull('notificacion_vencimiento_enviada')
            ->orWhere(function($query) use ($fecha_limite) {
                $query->where('notificacion_vencimiento_enviada', '<', Carbon::now()->subDays(1))
                      ->where('fecha_fin', '<=', $fecha_limite)
                      ->where('fecha_fin', '>=', Carbon::now());
            })
            ->get();

        $this->info("Se encontraron {$ofertas_por_vencer->count()} ofertas por vencer");

        foreach ($ofertas_por_vencer as $oferta) {
            $this->procesarOfertaPorVencer($oferta, $days);
        }

        // Verificar ofertas que ya vencieron hoy
        $ofertas_vencidas = Oferta::where('activo', true)
            ->whereDate('fecha_fin', Carbon::today())
            ->whereNull('notificacion_vencida_enviada')
            ->get();

        $this->info("Se encontraron {$ofertas_vencidas->count()} ofertas que vencieron hoy");

        foreach ($ofertas_vencidas as $oferta) {
            $this->procesarOfertaVencida($oferta);
        }

        // Verificar ofertas con bajo rendimiento
        $this->verificarOfertasBajoRendimiento();

        $this->info('Proceso completado exitosamente');
    }

    private function procesarOfertaPorVencer(Oferta $oferta, int $days)
    {
        $dias_restantes = Carbon::now()->diffInDays($oferta->fecha_fin, false);

        $this->info("Procesando oferta: {$oferta->nombre} (vence en {$dias_restantes} días)");

        // Obtener usuarios administrativos de la empresa
        $usuarios = User::where('empresa_id', $oferta->empresa_id)
            ->where('status', true)
            ->whereHas('roles', function($query) {
                $query->whereIn('name', ['admin', 'manager', 'super-admin']);
            })
            ->get();

        foreach ($usuarios as $usuario) {
            // Enviar notificación por WhatsApp
            $this->enviarNotificacionWhatsApp($usuario, $oferta, 'por_vencer', $dias_restantes);

            // Enviar notificación por email
            $this->enviarNotificacionEmail($usuario, $oferta, 'por_vencer', $dias_restantes);

            // Crear notificación en el sistema
            $this->crearNotificacionSistema($usuario, $oferta, 'por_vencer', $dias_restantes);
        }

        // Marcar que se envió la notificación
        $oferta->notificacion_vencimiento_enviada = Carbon::now();
        $oferta->save();

        $this->info("Notificaciones enviadas para la oferta: {$oferta->nombre}");
    }

    private function procesarOfertaVencida(Oferta $oferta)
    {
        $this->info("Procesando oferta vencida: {$oferta->nombre}");

        // Obtener usuarios administrativos
        $usuarios = User::where('empresa_id', $oferta->empresa_id)
            ->where('status', true)
            ->whereHas('roles', function($query) {
                $query->whereIn('name', ['admin', 'manager', 'super-admin']);
            })
            ->get();

        foreach ($usuarios as $usuario) {
            // Enviar notificaciones
            $this->enviarNotificacionWhatsApp($usuario, $oferta, 'vencida');
            $this->enviarNotificacionEmail($usuario, $oferta, 'vencida');
            $this->crearNotificacionSistema($usuario, $oferta, 'vencida');
        }

        // Marcar que se envió la notificación de vencimiento
        $oferta->notificacion_vencida_enviada = Carbon::now();
        $oferta->activo = false; // Desactivar la oferta automáticamente
        $oferta->save();

        $this->info("Oferta desactivada y notificaciones enviadas: {$oferta->nombre}");
    }

    private function verificarOfertasBajoRendimiento()
    {
        $this->info("Verificando ofertas con bajo rendimiento...");

        // Ofertas activas con menos de 5 usos en los últimos 30 días
        $ofertas_bajo_rendimiento = Oferta::where('activo', true)
            ->where('fecha_inicio', '<=', Carbon::now()->subDays(30))
            ->where('usos_actuales', '<', 5)
            ->where(function($query) {
                $query->whereNull('notificacion_bajo_rendimiento_enviada')
                      ->orWhere('notificacion_bajo_rendimiento_enviada', '<', Carbon::now()->subDays(7));
            })
            ->get();

        foreach ($ofertas_bajo_rendimiento as $oferta) {
            $this->procesarOfertaBajoRendimiento($oferta);
        }
    }

    private function procesarOfertaBajoRendimiento(Oferta $oferta)
    {
        $this->info("Procesando oferta con bajo rendimiento: {$oferta->nombre} ({$oferta->usos_actuales} usos)");

        // Obtener usuarios administrativos
        $usuarios = User::where('empresa_id', $oferta->empresa_id)
            ->where('status', true)
            ->whereHas('roles', function($query) {
                $query->whereIn('name', ['admin', 'manager', 'super-admin']);
            })
            ->get();

        foreach ($usuarios as $usuario) {
            // Enviar notificaciones con sugerencias
            $this->enviarNotificacionWhatsApp($usuario, $oferta, 'bajo_rendimiento');
            $this->enviarNotificacionEmail($usuario, $oferta, 'bajo_rendimiento');
            $this->crearNotificacionSistema($usuario, $oferta, 'bajo_rendimiento');
        }

        // Marcar que se envió la notificación de bajo rendimiento
        $oferta->notificacion_bajo_rendimiento_enviada = Carbon::now();
        $oferta->save();
    }

    private function enviarNotificacionWhatsApp($usuario, $oferta, $tipo, $dias_restantes = null)
    {
        try {
            $mensajes = [
                'por_vencer' => "⚠️ *Oferta por vencer*\n\nLa oferta '{$oferta->nombre}' vence en {$dias_restantes} días.\n\n📊 Usos actuales: {$oferta->usos_actuales}\n📅 Fecha fin: {$oferta->fecha_fin->format('d/m/Y')}\n\nRecomendación: Extender la fecha o crear una nueva oferta.",

                'vencida' => "❌ *Oferta vencida*\n\nLa oferta '{$oferta->nombre}' ha vencido el día de hoy.\n\n📊 Usos totales: {$oferta->usos_actuales}\n📅 Fecha vencimiento: {$oferta->fecha_fin->format('d/m/Y')}\n\nLa oferta ha sido desactivada automáticamente.",

                'bajo_rendimiento' => "📉 *Oferta con bajo rendimiento*\n\nLa oferta '{$oferta->nombre}' ha tenido solo {$oferta->usos_actuales} usos en los últimos 30 días.\n\n💡 Sugerencias:\n• Revisar el valor del descuento\n• Mejorar la promoción\n• Considerar cambiar los productos incluidos\n• Extender el período de vigencia"
            ];

            if (isset($mensajes[$tipo])) {
                // Aquí se integraría con el sistema de WhatsApp existente
                // Por ahora solo registramos el intento
                $this->info("WhatsApp pendiente para {$usuario->name}: {$mensajes[$tipo]}");
            }
        } catch (\Exception $e) {
            $this->error("Error enviando WhatsApp: " . $e->getMessage());
        }
    }

    private function enviarNotificacionEmail($usuario, $oferta, $tipo, $dias_restantes = null)
    {
        try {
            $asuntos = [
                'por_vencer' => "Oferta por vencer: {$oferta->nombre}",
                'vencida' => "Oferta vencida: {$oferta->nombre}",
                'bajo_rendimiento' => "Oferta con bajo rendimiento: {$oferta->nombre}"
            ];

            $this->info("Email pendiente para {$usuario->email} - Asunto: {$asuntos[$tipo]}");

            // Aquí se integraría con el sistema de email de Laravel
            // Mail::to($usuario->email)->send(new OfferNotificationMail($oferta, $tipo, $dias_restantes));

        } catch (\Exception $e) {
            $this->error("Error enviando email: " . $e->getMessage());
        }
    }

    private function crearNotificacionSistema($usuario, $oferta, $tipo, $dias_restantes = null)
    {
        try {
            $notificacion = \App\Models\Notification::create([
                'user_id' => $usuario->id,
                'type' => 'offer_notification',
                'title' => $this->getTituloNotificacion($tipo, $oferta, $dias_restantes),
                'message' => $this->getMensajeNotificacion($tipo, $oferta, $dias_restantes),
                'data' => json_encode([
                    'oferta_id' => $oferta->id,
                    'tipo' => $tipo,
                    'dias_restantes' => $dias_restantes,
                    'action_url' => route('admin.ofertas.edit', $oferta->id)
                ]),
                'read_at' => null,
                'created_at' => Carbon::now()
            ]);

            $this->info("Notificación creada para {$usuario->name}");
        } catch (\Exception $e) {
            $this->error("Error creando notificación: " . $e->getMessage());
        }
    }

    private function getTituloNotificacion($tipo, $oferta, $dias_restantes = null)
    {
        $titulos = [
            'por_vencer' => "Oferta por vencer: {$oferta->nombre}",
            'vencida' => "Oferta vencida: {$oferta->nombre}",
            'bajo_rendimiento' => "Oferta con bajo rendimiento: {$oferta->nombre}"
        ];

        return $titulos[$tipo] ?? 'Notificación de oferta';
    }

    private function getMensajeNotificacion($tipo, $oferta, $dias_restantes = null)
    {
        $mensajes = [
            'por_vencer' => "La oferta '{$oferta->nombre}' vence en {$dias_restantes} días. Usos actuales: {$oferta->usos_actuales}",
            'vencida' => "La oferta '{$oferta->nombre}' ha vencido. Usos totales: {$oferta->usos_actuales}",
            'bajo_rendimiento' => "La oferta '{$oferta->nombre}' ha tenido solo {$oferta->usos_actuales} usos en los últimos 30 días."
        ];

        return $mensajes[$tipo] ?? 'Notificación de oferta';
    }
}
