<?php

require_once 'vendor/autoload.php';

$app = require_once 'bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

echo "🔍 EVALUACIÓN COMPLETA DE LA API WHATSAPP\n";
echo "==========================================\n\n";

try {
    $whatsappService = new \App\Services\WhatsAppService();
    
    // 1. Conectividad y Estado
    echo "1. 🌐 CONECTIVIDAD Y ESTADO\n";
    echo "   -------------------------\n";
    
    $connection = $whatsappService->testConnection();
    echo "   Conexión API: " . ($connection['success'] ? '✅ OK' : '❌ FAIL') . "\n";
    
    $status = $whatsappService->getStatus();
    echo "   Estado WhatsApp: " . ($status['success'] ? '✅ OK' : '❌ FAIL') . "\n";
    if (isset($status['status'])) {
        echo "   Estado específico: " . $status['status'] . "\n";
        echo "   Listo para enviar: " . ($status['isReady'] ? '✅ SÍ' : '❌ NO') . "\n";
    }
    echo "\n";
    
    // 2. Autenticación JWT
    echo "2. 🔐 AUTENTICACIÓN JWT\n";
    echo "   -------------------\n";
    
    $configFile = base_path('whatsapp-service-config.env');
    if (file_exists($configFile)) {
        $content = file_get_contents($configFile);
        $hasToken = strpos($content, 'LARAVEL_API_KEY=') !== false && 
                   !empty(trim(substr($content, strpos($content, 'LARAVEL_API_KEY=') + 16)));
        echo "   Token JWT configurado: " . ($hasToken ? '✅ SÍ' : '❌ NO') . "\n";
        
        if ($hasToken) {
            // Probar verificación JWT
            $response = \Illuminate\Support\Facades\Http::withHeaders([
                'Authorization' => 'Bearer ' . trim(substr($content, strpos($content, 'LARAVEL_API_KEY=') + 16)),
                'Accept' => 'application/json',
            ])->post('http://localhost:8000/api/verify-token');
            
            echo "   Verificación JWT: " . ($response->successful() ? '✅ VÁLIDO' : '❌ INVÁLIDO') . "\n";
        }
    } else {
        echo "   Archivo de configuración: ❌ NO EXISTE\n";
    }
    echo "\n";
    
    // 3. Funcionalidades Core
    echo "3. ⚙️ FUNCIONALIDADES CORE\n";
    echo "   ----------------------\n";
    
    // Test envío de mensaje
    $testResult = $whatsappService->sendMessage('584241234567', '🧪 Test API evaluation');
    echo "   Envío de mensajes: " . ($testResult['success'] ? '✅ OK' : '❌ FAIL') . "\n";
    if (!$testResult['success']) {
        echo "      Error: " . $testResult['error'] . "\n";
    }
    
    // Test obtener QR
    $qrResult = $whatsappService->getQR();
    echo "   Obtener QR: " . ($qrResult['success'] ? '✅ OK' : '❌ FAIL') . "\n";
    
    echo "\n";
    
    // 4. Configuración y Optimización
    echo "4. ⚡ CONFIGURACIÓN Y OPTIMIZACIÓN\n";
    echo "   --------------------------------\n";
    
    $config = config('whatsapp');
    echo "   Timeouts configurados: " . (isset($config['timeouts']) ? '✅ SÍ' : '❌ NO') . "\n";
    if (isset($config['timeouts'])) {
        echo "      - Conexión: " . ($config['timeouts']['connection'] ?? 'N/A') . "s\n";
        echo "      - Estado: " . ($config['timeouts']['status'] ?? 'N/A') . "s\n";
        echo "      - Envío: " . ($config['timeouts']['send_message'] ?? 'N/A') . "s\n";
    }
    
    echo "   Cache configurado: " . (isset($config['cache']) ? '✅ SÍ' : '❌ NO') . "\n";
    echo "\n";
    
    // 5. Jobs y Procesamiento Asíncrono
    echo "5. 🔄 JOBS Y PROCESAMIENTO ASÍNCRONO\n";
    echo "   ----------------------------------\n";
    
    $jobExists = class_exists('\App\Jobs\SendWelcomeWhatsAppMessage');
    echo "   Job de bienvenida: " . ($jobExists ? '✅ EXISTE' : '❌ NO EXISTE') . "\n";
    
    // Verificar si hay jobs pendientes
    try {
        $pendingJobs = \DB::table('jobs')->count();
        echo "   Jobs pendientes: {$pendingJobs}\n";
    } catch (\Exception $e) {
        echo "   Jobs pendientes: ❌ NO SE PUEDE VERIFICAR\n";
    }
    
    echo "\n";
    
    // 6. Recomendaciones de Mejora
    echo "6. 💡 RECOMENDACIONES DE MEJORA\n";
    echo "   -----------------------------\n";
    
    $recommendations = [];
    
    if (!$connection['success']) {
        $recommendations[] = "🔧 Verificar que el servidor Node.js esté ejecutándose en puerto 3000";
    }
    
    if (!isset($status['isReady']) || !$status['isReady']) {
        $recommendations[] = "📱 Escanear código QR para conectar WhatsApp Web";
    }
    
    if (!$hasToken ?? false) {
        $recommendations[] = "🔑 Configurar token JWT ejecutando: php generate-simple-jwt.php";
    }
    
    if (!isset($config['cache'])) {
        $recommendations[] = "⚡ Configurar cache para mejorar rendimiento";
    }
    
    try {
        $queueConnection = config('queue.default');
        if ($queueConnection === 'sync') {
            $recommendations[] = "🔄 Cambiar queue driver a 'database' o 'redis' para mejor rendimiento";
        }
    } catch (\Exception $e) {
        $recommendations[] = "🔄 Configurar sistema de colas para procesamiento asíncrono";
    }
    
    if (empty($recommendations)) {
        echo "   ✅ ¡La API está bien configurada!\n";
    } else {
        foreach ($recommendations as $i => $rec) {
            echo "   " . ($i + 1) . ". {$rec}\n";
        }
    }
    
    echo "\n";
    
    // 7. Puntuación General
    echo "7. 📊 PUNTUACIÓN GENERAL\n";
    echo "   ---------------------\n";
    
    $score = 0;
    $maxScore = 10;
    
    if ($connection['success']) $score += 2;
    if (isset($status['isReady']) && $status['isReady']) $score += 2;
    if ($hasToken ?? false) $score += 2;
    if ($testResult['success']) $score += 2;
    if (isset($config['timeouts'])) $score += 1;
    if ($jobExists) $score += 1;
    
    $percentage = ($score / $maxScore) * 100;
    
    echo "   Puntuación: {$score}/{$maxScore} ({$percentage}%)\n";
    
    if ($percentage >= 90) {
        echo "   Estado: 🟢 EXCELENTE\n";
    } elseif ($percentage >= 70) {
        echo "   Estado: 🟡 BUENO\n";
    } elseif ($percentage >= 50) {
        echo "   Estado: 🟠 REGULAR\n";
    } else {
        echo "   Estado: 🔴 NECESITA MEJORAS\n";
    }
    
} catch (Exception $e) {
    echo "❌ Error durante evaluación: " . $e->getMessage() . "\n";
}

echo "\n==========================================\n";
echo "Evaluación completada ✅\n";